﻿//DESCRIPTION:Build angled table heads

// Modified 2022-01-30
// Keith Gilbert, Gilbert Consulting
// www.gilbertconsulting.com

#targetengine "main"

Pre();

function Pre () {
    // Is there a file open?
	if (!app.documents.length) {
		alert("No InDesign documents are open. Please open a document and try again.");
		return;
	}  
    // Is there a selection?
    if (app.selection.length != 1) {
		alert("Please select a table and try again.");
		return;
	}  
    var mySelection = app.selection[0];
    // Is it a table selection?
    switch(mySelection.constructor.name){
        // When a row, a column, or a range of cells is selected, the type returned is "Cell"
        case "Cell":
            var myTable = mySelection.parent;
            app.select(mySelection.cells[0].insertionPoints[0]); // This is necessary because of a bug: you can't reset the proxy to upper-left via script when a cell is selected
            break;
        case "Table":
            var myTable = mySelection;
            app.select(mySelection.rows[0].cells[0].insertionPoints[0]); // This is necessary because of a bug: you can't reset the proxy to upper-left via script when a cell is selected
            break;
        case "InsertionPoint":
        case "Character":
        case "Word":
        case "TextStyleRange":
        case "Line":
        case "Paragraph":
        case "TextColumn":
        case "Text":
            if(mySelection.parent.constructor.name == "Cell"){
                var myTable = mySelection.parent.parent;
            }
            else {
                alert("Please select a table and try again.");
                return;
            }
            break;
        case "Rectangle":
        case "Oval":
        case "Polygon":
        case "GraphicLine":
        case "TextFrame":
            if(mySelection.parent.parent.constructor.name == "Cell"){
                var myTable = mySelection.parent.parent.parent;
            }
            else {
                alert("Please select a table and try again.");
                return;
            }
            break;
        case "Image":
        case "PDF":
        case "EPS":
            if(mySelection.parent.parent.parent.constructor.name == "Cell"){
                var myTable = mySelection.parent.parent.parent.parent;
            }
            else {
                alert("Please select a table and try again.");
                return;
            }
            break;
        default:
            break;
    }
    if (myTable.isValid) {
        HireMe();
        app.doScript('Main(myTable);', ScriptLanguage.JAVASCRIPT, [myTable], UndoModes.ENTIRE_SCRIPT,"Run Script"); // This makes the script un-doable
        // Main(myTable);     
    }
    else {
        alert("Please select a table and try again.");
        return;
    }
}
function Main(myTable) {
    // Get some parameters from the user
	var myParameters = myGetParameters();
	if (myParameters.length == 0) {
		return;
	}    
    if (myParameters[0] == true) { // 15°
        var myAngle = 15;
        var myConstant = 1.0353;
    }
    else {
        if (myParameters[1] == true) { // 30°
            var myAngle = 30;
            var myConstant = 1.1547;
        }
        else {
            if (myParameters[2] == true) { // 45°
                var myAngle = 45;
                var myConstant = 1.4142;
            }
            else {
                if (myParameters[3] == true) { // 60°
                    var myAngle = 60;
                    var myConstant = 2;
                }
                else {
                    if (myParameters[4] == true) { // -15
                        var myAngle = -15;
                        var myConstant = 1.0353;
                    }
                    else {
                        if (myParameters[5] == true) { // -30
                            var myAngle = -30;
                            var myConstant = 1.1547;
                        }
                        else {
                            if (myParameters[6] == true) { // -45
                                var myAngle = -45;
                                var myConstant = 1.4142;
                            }
                            else {
                                if (myParameters[7] == true) { // -60
                                    var myAngle = -60;
                                    var myConstant = 2;
                                }
                            }
                        }
                    }
                }
            }
        }
    } 
    var myCopyTypeSpecs = myParameters[8];
    var myApplyStyle = myParameters[9]
    var myExtract = myParameters[10];
    var myRemove = myParameters[11];
    var myDelete = myParameters[12];
    var myPreserveStrokes = myParameters[13];
    var myAnchor = myParameters[14];
    var myHeight = Number(myParameters[15]);
    var  
		_scriptRedraw = app.scriptPreferences.enableRedraw,  
		_userInteraction = app.scriptPreferences.userInteractionLevel;  
		_measurementUnit = app.scriptPreferences.measurementUnit;
        _transformReferencePoint = app.activeWindow.transformReferencePoint;
	app.scriptPreferences.enableRedraw = false;  
	app.scriptPreferences.userInteractionLevel = UserInteractionLevels.neverInteract;  
	app.scriptPreferences.measurementUnit = MeasurementUnits.POINTS;
    app.activeWindow.transformReferencePoint = AnchorPoint.TOP_LEFT_ANCHOR;
    // Store the text from row 1
    if (myExtract) { // User chose to extract the header text from row 1 of the table
        var myRow1Array = new Array;
        for (var i=0; i<myTable.rows[0].cells.length; i++) {
            var myString = myTable.rows[0].cells[i].texts[0].contents;
            if (myRemove) { // User chose to remove line breaks/returns from the extracted text
                myString = myString.replace(/\s*(\n|\r)\s*/g," ");
            }
            myRow1Array.push(myString);
        }
    }
    // Copy type specs from row 1, column 1
    if (myCopyTypeSpecs) { // User chose to copy the type specs from row 1, column 1
        var myTypeProps = myTable.rows[0].cells[0].insertionPoints[0].properties;
    }
    // Find the coordinates of the upper-left table corner
    var myCoordinates = myGetCoordinates(myTable);
    var myX = myCoordinates[0];
    var myY = myCoordinates[1];
    var myNumColumns = myTable.columnCount;
    var myWidth = myTable.width/myConstant;
    myHeight = myHeight*myConstant;
    var myDoc = app.activeDocument;
    if (myTable.parent.parentPage == null) { // Table is on the pasteboard
        var myPage = myTable.parent.parent;
    }
    else {
        var myPage = myTable.parent.parentPage;
    }
    var myTF = myPage.textFrames.add();
    // Create the text frame
    myTF.geometricBounds = [myY,myX,myY+myWidth,myX+myHeight];
    myTF.textFramePreferences.textColumnCount = 1;
    myTF.textFramePreferences.insetSpacing = [0,0,0,0];
    myTF.textFramePreferences.verticalJustification = VerticalJustification.TOP_ALIGN;
    myTF.textFramePreferences.autoSizingType = AutoSizingTypeEnum.OFF;
    myTF.absoluteRotationAngle = 90-myAngle;
    myTF.absoluteShearAngle = 0-myAngle;
    // Add the table
    try {
        var myHeaderTable = myTF.tables.add({
            headerRowCount : 0,
            bodyRowCount   : myNumColumns,
            columnCount    : 1,
            width          : myHeight,
            height         : myWidth });
    }
    catch (myError) {
        alert("Unable to create headers. Please carefully check the value you entered for the width and try again. The script will now exit.");
        app.scriptPreferences.enableRedraw = _scriptRedraw;  
        app.scriptPreferences.userInteractionLevel = _userInteraction;  
        app.scriptPreferences.measurementUnit = _measurementUnit;
        app.activeWindow.transformReferencePoint = _transformReferencePoint;
        return;
    }
    // Get the character that holds the table
    var myTableCharacter = resolve(
        myHeaderTable.            // Table
        storyOffset.    // InsertionPoint
        toSpecifier().    // contains "insertion-point" once
        replace('insertion-point','character')
        );
    var myTableIP = myTableCharacter.insertionPoints[0];
    myTableIP.firstLineIndent = 0;
    myTableIP.leftIndent = 0;
    // Populate the cells and set text and cell attributes
    for (var i=0; i<myHeaderTable.bodyRowCount; i++) {
        var mySourceColWidth = myTable.columns[i].width;
        myHeaderTable.rows[i].height = mySourceColWidth/myConstant;
        myHeaderTable.rows[i].autoGrow = false;
        if (myCopyTypeSpecs) {
            myHeaderTable.rows[i].cells[0].texts[0].properties = myTypeProps;
        }
        else {
            myHeaderTable.rows[i].cells[0].texts[0].pointSize = 10;
        }
        if (myExtract) {
            if (myRow1Array.length > 0) {
                myHeaderTable.rows[i].cells[0].contents = myRow1Array.shift();
            }
            else {
             myHeaderTable.rows[i].cells[0].contents = "Xxxxxx";
            }
        }
        else {
            myHeaderTable.rows[i].cells[0].contents = "Xxxxxx";
        }
        myHeaderTable.rows[i].verticalJustification = VerticalJustification.CENTER_ALIGN
        myHeaderTable.rows[i].leftInset = 4*myConstant;
        myHeaderTable.rows[i].rightInset = 0;
        myHeaderTable.rows[i].topInset = 0;
        myHeaderTable.rows[i].bottomInset = 0;
        myHeaderTable.rows[i].rightEdgeStrokeWeight = 0;
        myHeaderTable.rows[i].leftEdgeStrokeWeight = 0;
        if ((myTable.columns[i].cells[0].leftEdgeStrokeWeight != 0) && 
            (myTable.columns[i].cells[0].leftEdgeStrokeType != null) &&
            (myTable.columns[i].cells[0].leftEdgeStrokeColor != myDoc.swatches.item("None"))) {
            if (myPreserveStrokes) {
                myHeaderTable.rows[i].topEdgeStrokeWeight = myTable.columns[i].cells[0].leftEdgeStrokeWeight;
            }
            else {
                myHeaderTable.rows[i].topEdgeStrokeWeight = myTable.columns[i].cells[0].leftEdgeStrokeWeight/myConstant;
            }
            myHeaderTable.rows[i].topEdgeStrokeType = myTable.columns[i].cells[0].leftEdgeStrokeType;
            myHeaderTable.rows[i].topEdgeStrokeTint = myTable.columns[i].cells[0].leftEdgeStrokeTint;
            myHeaderTable.rows[i].topEdgeStrokeColor = myTable.columns[i].cells[0].leftEdgeStrokeColor;
            myHeaderTable.rows[i].topEdgeStrokeGapTint = myTable.columns[i].cells[0].leftEdgeStrokeGapTint;
            myHeaderTable.rows[i].topEdgeStrokeGapColor = myTable.columns[i].cells[0].leftEdgeStrokeGapColor;
        }
        else {
            myHeaderTable.rows[i].topEdgeStrokeWeight = 0;
        }
    }
    var mySourceRightStrokeWeight = myTable.columns[-1].cells[0].rightEdgeStrokeWeight;
    if (myTable.columns[-1].cells[0].rightEdgeStrokeType != null) {
        if (myPreserveStrokes) {
            myHeaderTable.rows[-1].bottomEdgeStrokeWeight = mySourceRightStrokeWeight;
        }
        else {
            myHeaderTable.rows[-1].bottomEdgeStrokeWeight = mySourceRightStrokeWeight/myConstant;
        }
        myHeaderTable.rows[-1].bottomEdgeStrokeType = myTable.columns[-1].cells[0].leftEdgeStrokeType;
        myHeaderTable.rows[-1].bottomEdgeStrokeTint = myTable.columns[-1].cells[0].leftEdgeStrokeTint;
        myHeaderTable.rows[-1].bottomEdgeStrokeColor = myTable.columns[-1].cells[0].leftEdgeStrokeColor;
        myHeaderTable.rows[-1].bottomEdgeStrokeGapTint = myTable.columns[-1].cells[0].leftEdgeStrokeGapTint;
        myHeaderTable.rows[-1].bottomEdgeStrokeGapColor = myTable.columns[-1].cells[0].leftEdgeStrokeGapColor;
    }
    else {
        myHeaderTable.rows[-1].bottomEdgeStrokeWeight = 0;
    }
    myHeaderTable.rows.everyItem().cells.everyItem().texts[0].justification = Justification.leftAlign;
    myHeaderTable.rows.everyItem().cells.everyItem().texts[0].skew = myAngle;
    // If the rotation angle is negative, rotate the table cell contents and right align the text
    if (myAngle < 0) {
        myHeaderTable.rows.everyItem().cells.everyItem().rotationAngle = 180;
        myHeaderTable.rows.everyItem().cells.everyItem().texts[0].justification = Justification.rightAlign;
    }
    if (myApplyStyle) {
        if (!locateParagraphStyle("Angled table head")) { // Check to see if the style exists, possibly in a style group
            var myParagraphStyle = myDoc.paragraphStyles.add({name:"Angled table head"});
            myParagraphStyle.properties = myHeaderTable.rows[0].cells[0].texts[0].properties;
        }
        myHeaderTable.rows.everyItem().cells.everyItem().texts[0].appliedParagraphStyle = myDoc.paragraphStyles.itemByID(locateParagraphStyle("Angled table head"));
    }
    myTF.fit(FitOptions.FRAME_TO_CONTENT);
    // Delete the first row of the table if option is selected
    if (myDelete) {
        myTable.rows[0].remove();
    }
    // Anchor the table to the upper-left cell if option is selected
    if (myAnchor) {
        // Anchor the diagonal header group in the text frame
        myTF.anchoredObjectSettings.insertAnchoredObject(myTable.rows[0].cells[0].insertionPoints[0]);
    }
    app.scriptPreferences.enableRedraw = _scriptRedraw;  
	app.scriptPreferences.userInteractionLevel = _userInteraction;  
	app.scriptPreferences.measurementUnit = _measurementUnit;
    app.activeWindow.transformReferencePoint = _transformReferencePoint;
}
// Locate the upper-left coordinate of a table
function myGetCoordinates(myTable) {
    // Find the y coordinate
    var myCell = myTable.rows[0].cells[0];
    var tI = myCell.topInset;
    var fB = myCell.firstBaselineOffset;
    var vJ = myCell.verticalJustification;
    myCell.firstBaselineOffset = FirstBaseline.fixedHeight;
    myCell.minimumFirstBaselineOffset = 0;
    myCell.verticalJustification = VerticalJustification.topAlign;
    var myY = myCell.texts[0].insertionPoints[0].baseline - tI - (myCell.topEdgeStrokeWeight/2);
    myCell.verticalJustification = vJ;
    myCell.firstBaselineOffset = fB;
    // Find the x coordinate
    // Get the character that holds the table
    var myCharacter = resolve(
        myTable.            // Table
        storyOffset.    // InsertionPoint
        toSpecifier().    // contains "insertion-point" once
        replace('insertion-point','character')
        );
    var myTF = myCharacter.parentTextFrames[0];
    var myX = myCharacter.horizontalOffset+1;
    return[myX,myY];
}
// Get parameters
function myGetParameters() {
    var myUnitsText = app.activeDocument.viewPreferences.verticalMeasurementUnits.toString().toLowerCase();
    switch (app.activeDocument.viewPreferences.verticalMeasurementUnits.toString()){
        case "POINTS":
            var myMultiplier = 1;
            break;
        case "PICAS":
            var myMultiplier = 6;
            break;
        case "INCHES":
            var myMultiplier = 72;
            break;
        case "INCHES_DECIMAL":
            var myMultiplier = 72;
            break;
        case "MILLIMETERS":
            var myMultiplier = 2.8346;
            break;
       case "CENTIMETERS":
            var myMultiplier = 28.346;
            break;
        case "CICEROS":
            var myMultiplier = 12.789;
            break;
        case "AGATES":
            var myMultiplier = 5.143;
            break;
        case "PIXELS":
            var myMultiplier = 1;
            break;
        default:
            break;
    }

    /*
    Code for Import https://scriptui.joonas.me — (Triple click to select): 
    {"activeId":16,"items":{"item-0":{"id":0,"type":"Dialog","parentId":false,"style":{"varName":"w","text":"Angle table heads","preferredSize":[0,0],"margins":20,"orientation":"column","spacing":10,"alignChildren":["center","top"],"windowType":"Dialog","creationProps":{"su1PanelCoordinates":false,"maximizeButton":false,"minimizeButton":false,"independent":false,"closeButton":true,"borderless":false,"resizeable":false},"enabled":true}},"item-2":{"id":2,"type":"Button","parentId":5,"style":{"varName":"Cancel","text":"Cancel","justify":"center","preferredSize":[70,0],"alignment":null,"helpTip":null,"enabled":true}},"item-3":{"id":3,"type":"Button","parentId":5,"style":{"varName":"Ok","text":"OK","justify":"center","preferredSize":[70,0],"alignment":null,"helpTip":null,"enabled":true}},"item-5":{"id":5,"type":"Group","parentId":0,"style":{"varName":null,"preferredSize":[0,0],"margins":[15,0,0,0],"orientation":"row","spacing":10,"alignChildren":["right","center"],"alignment":"right","enabled":true}},"item-9":{"id":9,"type":"Group","parentId":0,"style":{"enabled":true,"varName":null,"preferredSize":[0,0],"margins":0,"orientation":"row","spacing":10,"alignChildren":["left","top"],"alignment":null}},"item-12":{"id":12,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"my15","text":"15°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-13":{"id":13,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"my45","text":"45°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":true}},"item-15":{"id":15,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"my30","text":"30°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-16":{"id":16,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"my60","text":"60°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-17":{"id":17,"type":"Panel","parentId":29,"style":{"enabled":true,"varName":null,"creationProps":{"borderStyle":"etched","su1PanelCoordinates":false},"text":"Text angle","preferredSize":[0,0],"margins":20,"orientation":"column","spacing":15,"alignChildren":["left","top"],"alignment":null}},"item-19":{"id":19,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myExtract","text":"Extract table head text from row 1","preferredSize":[0,0],"alignment":null,"helpTip":"Select this if the text you want to appear in the angled table heads is in row 1 of your table. Otherwise, placeholder text will be used for the angled table heads."}},"item-20":{"id":20,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myDelete","text":"Delete row 1","preferredSize":[0,0],"alignment":null,"helpTip":"Delete the first row of the table after the diagonal heads are created."}},"item-21":{"id":21,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myAnchor","text":"Anchor diagonal heads to cell 1","preferredSize":[0,0],"alignment":null,"helpTip":"Anchor the table that contains the angled table heads to the upper-left cell of the table below the heads."}},"item-24":{"id":24,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myRemove","text":"Remove line breaks/returns from extracted text","preferredSize":[0,0],"alignment":null,"helpTip":"Remove any line breaks and returns from the text that is extracted from row 1 to build the table head text."}},"item-25":{"id":25,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myCopyTypeSpecs","text":"Copy type specs from the first cell of row 1","preferredSize":[0,0],"alignment":null,"helpTip":"Copy the type specifications from the left-most cell of row 1 and use these specs for the text in the table head."}},"item-26":{"id":26,"type":"Group","parentId":31,"style":{"enabled":true,"varName":null,"preferredSize":[0,0],"margins":[0,0,0,0],"orientation":"row","spacing":10,"alignChildren":["left","center"],"alignment":null}},"item-27":{"id":27,"type":"StaticText","parentId":26,"style":{"enabled":true,"varName":null,"creationProps":{"truncate":"none","multiline":false,"scrolling":false},"softWrap":false,"text":"Height:","justify":"left","preferredSize":[0,0],"alignment":null,"helpTip":"The height that you want set aside for the table heads. The units are extracted from the vertical measurement system of the document."}},"item-28":{"id":28,"type":"EditText","parentId":26,"style":{"enabled":true,"varName":"myHeight","creationProps":{"noecho":false,"readonly":false,"multiline":false,"scrollable":false,"borderless":false,"enterKeySignalsOnChange":false},"softWrap":false,"text":"","justify":"left","preferredSize":[60,0],"alignment":null,"helpTip":null}},"item-29":{"id":29,"type":"Group","parentId":9,"style":{"enabled":true,"varName":null,"preferredSize":[0,0],"margins":0,"orientation":"column","spacing":10,"alignChildren":["left","center"],"alignment":null}},"item-30":{"id":30,"type":"StaticText","parentId":26,"style":{"enabled":true,"varName":"myMeasurementUnits","creationProps":{"truncate":"none","multiline":false,"scrolling":false},"softWrap":false,"text":"\"myUnitsText\"","justify":"left","preferredSize":[0,0],"alignment":null,"helpTip":null}},"item-31":{"id":31,"type":"Group","parentId":9,"style":{"enabled":true,"varName":null,"preferredSize":[0,0],"margins":[0,0,0,10],"orientation":"column","spacing":15,"alignChildren":["left","center"],"alignment":null}},"item-32":{"id":32,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myApplyStyle","text":"Apply paragraph style \"Angled table head\" to text","preferredSize":[0,0],"alignment":null,"helpTip":"Create and apply a paragraph style named \"Angled table head\" to the angled text. If the paragraph style already exists, the existing style will be used, even if it contains incorrect formatting."}},"item-33":{"id":33,"type":"Checkbox","parentId":31,"style":{"enabled":true,"varName":"myPreserveStrokes","text":"Preserve original stroke thickness","preferredSize":[0,0],"alignment":null,"helpTip":"Leave this option unchecked if you want the strokes in the table head to butt up precisely with the strokes in the body of the table."}},"item-34":{"id":34,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"myNeg15","text":"-15","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-35":{"id":35,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"myNeg30","text":"-30°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-36":{"id":36,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"myNeg45","text":"-45°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}},"item-37":{"id":37,"type":"RadioButton","parentId":17,"style":{"enabled":true,"varName":"myNeg60","text":"-60°","preferredSize":[0,0],"alignment":null,"helpTip":null,"checked":false}}},"order":[0,9,29,17,12,15,13,16,34,35,36,37,31,26,27,28,30,25,32,19,24,20,33,21,5,2,3],"settings":{"importJSON":true,"indentSize":false,"cepExport":false,"includeCSSJS":true,"showDialog":false,"functionWrapper":false,"compactCode":false,"afterEffectsDockable":false,"itemReferenceList":"None"}}
    */ 

    // W
    // =
    var w = new Window("dialog"); 
        w.text = "Angle table heads"; 
        w.orientation = "column"; 
        w.alignChildren = ["center","top"]; 
        w.spacing = 10; 
        w.margins = 20; 

    // GROUP1
    // ======
    var group1 = w.add("group", undefined, {name: "group1"}); 
        group1.orientation = "row"; 
        group1.alignChildren = ["left","top"]; 
        group1.spacing = 10; 
        group1.margins = 0; 

    // GROUP2
    // ======
    var group2 = group1.add("group", undefined, {name: "group2"}); 
        group2.orientation = "column"; 
        group2.alignChildren = ["left","center"]; 
        group2.spacing = 10; 
        group2.margins = 0; 

    // PANEL1
    // ======
    var panel1 = group2.add("panel", undefined, undefined, {name: "panel1"}); 
        panel1.text = "Text angle"; 
        panel1.orientation = "column"; 
        panel1.alignChildren = ["left","top"]; 
        panel1.spacing = 15; 
        panel1.margins = 20; 

    var my15 = panel1.add("radiobutton", undefined, undefined, {name: "my15"}); 
        my15.text = "15°"; 

    var my30 = panel1.add("radiobutton", undefined, undefined, {name: "my30"}); 
        my30.text = "30°"; 

    var my45 = panel1.add("radiobutton", undefined, undefined, {name: "my45"}); 
        my45.text = "45°"; 
        my45.value = true; 

    var my60 = panel1.add("radiobutton", undefined, undefined, {name: "my60"}); 
        my60.text = "60°"; 

    var myNeg15 = panel1.add("radiobutton", undefined, undefined, {name: "myNeg15"}); 
        myNeg15.text = "-15"; 

    var myNeg30 = panel1.add("radiobutton", undefined, undefined, {name: "myNeg30"}); 
        myNeg30.text = "-30°"; 

    var myNeg45 = panel1.add("radiobutton", undefined, undefined, {name: "myNeg45"}); 
        myNeg45.text = "-45°"; 

    var myNeg60 = panel1.add("radiobutton", undefined, undefined, {name: "myNeg60"}); 
        myNeg60.text = "-60°"; 

    // GROUP3
    // ======
    var group3 = group1.add("group", undefined, {name: "group3"}); 
        group3.orientation = "column"; 
        group3.alignChildren = ["left","center"]; 
        group3.spacing = 15; 
        group3.margins = [10,0,0,0]; 

    // GROUP4
    // ======
    var group4 = group3.add("group", undefined, {name: "group4"}); 
        group4.orientation = "row"; 
        group4.alignChildren = ["left","center"]; 
        group4.spacing = 10; 
        group4.margins = [0,0,0,0]; 

    var statictext1 = group4.add("statictext", undefined, undefined, {name: "statictext1"}); 
        statictext1.helpTip = "The height that you want set aside for the table heads. The units are extracted from the vertical measurement system of the document."; 
        statictext1.text = "Height:"; 

    var myHeight = group4.add('edittext {properties: {name: "myHeight"}}'); 
        myHeight.preferredSize.width = 60; 

    var myMeasurementUnits = group4.add("statictext", undefined, undefined, {name: "myMeasurementUnits"}); 
        myMeasurementUnits.text = myUnitsText; 

    // GROUP3
    // ======
    var myCopyTypeSpecs = group3.add("checkbox", undefined, undefined, {name: "myCopyTypeSpecs"}); 
        myCopyTypeSpecs.helpTip = "Copy the type specifications from the left-most cell of row 1 and use these specs for the text in the table head."; 
        myCopyTypeSpecs.text = "Copy type specs from the first cell of row 1"; 

    var myApplyStyle = group3.add("checkbox", undefined, undefined, {name: "myApplyStyle"}); 
        myApplyStyle.helpTip = "Create and apply a paragraph style named \u0022Angled table head\u0022 to the angled text. If the paragraph style already exists, the existing style will be used, even if it contains incorrect formatting."; 
        myApplyStyle.text = "Apply paragraph style \u0022Angled table head\u0022 to text"; 

    var myExtract = group3.add("checkbox", undefined, undefined, {name: "myExtract"}); 
        myExtract.helpTip = "Select this if the text you want to appear in the angled table heads is in row 1 of your table. Otherwise, placeholder text will be used for the angled table heads."; 
        myExtract.text = "Extract table head text from row 1"; 

    var myRemove = group3.add("checkbox", undefined, undefined, {name: "myRemove"}); 
        myRemove.helpTip = "Remove any line breaks and returns from the text that is extracted from row 1 to build the table head text."; 
        myRemove.text = "Remove line breaks/returns from extracted text"; 

    var myDelete = group3.add("checkbox", undefined, undefined, {name: "myDelete"}); 
        myDelete.helpTip = "Delete the first row of the table after the diagonal heads are created."; 
        myDelete.text = "Delete row 1"; 

    var myPreserveStrokes = group3.add("checkbox", undefined, undefined, {name: "myPreserveStrokes"}); 
        myPreserveStrokes.helpTip = "Leave this option unchecked if you want the strokes in the table head to butt up precisely with the strokes in the body of the table."; 
        myPreserveStrokes.text = "Preserve original stroke thickness"; 

    var myAnchor = group3.add("checkbox", undefined, undefined, {name: "myAnchor"}); 
        myAnchor.helpTip = "Anchor the table that contains the angled table heads to the upper-left cell of the table below the heads."; 
        myAnchor.text = "Anchor diagonal heads to cell 1"; 

    // GROUP5
    // ======
    var group5 = w.add("group", undefined, {name: "group5"}); 
        group5.orientation = "row"; 
        group5.alignChildren = ["right","center"]; 
        group5.spacing = 10; 
        group5.margins = [0,15,0,0]; 
        group5.alignment = ["right","top"]; 

    var Cancel = group5.add("button", undefined, undefined, {name: "Cancel"}); 
        Cancel.text = "Cancel"; 
        Cancel.preferredSize.width = 70; 

    var Ok = group5.add("button", undefined, undefined, {name: "Ok"}); 
        Ok.text = "OK"; 
        Ok.preferredSize.width = 70; 


    ///////////// END PASTED SCRIPTUI CODE /////////////

    Ok.onClick = function() { // User clicked the OK button
        if ((myHeight.text <= 0) || isNaN(myHeight.text)) {
             alert("Please enter a valid number in the Height field.");
        }
        else {
            exit(); // This onClick function
        }
    }

    if (w.show() == 1) { // User didn't click the cancel button
        myHeight = Number(myHeight.text)*myMultiplier;
        return[my15.value, my30.value, my45.value, my60.value, myNeg15.value, myNeg30.value, myNeg45.value, myNeg60.value, myCopyTypeSpecs.value, myApplyStyle.value, myExtract.value, myRemove.value, myDelete.value, myPreserveStrokes.value, myAnchor.value, myHeight];
    }
    else {
        return[];
    }
}
// Display the "Hire-me-ware" dialog
function HireMe() {
	// Determine if the dialog needs to be displayed
	var logFile = File(myTrimPath($.fileName)+"/create_angled_table_heads_required.txt");
	var myElapsedTime = 0;
	if (logFile.exists) { // File exists, so see how much time has elapsed since the script was last run
		// Read the previous date from the log file
		logFile.open("r");
		var myPreviousDate = logFile.read();
		logFile.close();	
		var today = new Date();
		var myTimeOffset = today.getTime();
		// Compare the date with today's date
		myElapsedTime = today - myPreviousDate;
	}
	if ((myElapsedTime > 2592000000) || (!logFile.exists)) { // It has been more than 30 days since the script has run, or it has never been run
		// Get the path to the script ($.fileNames is core JS for the path and filename of the current script)
		var myHeadShot =  "\u00FF\u00D8\u00FF\u00E1\x00\x18Exif\x00\x00II*\x00\b\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\u00FF\u00EC\x00\x11Ducky\x00\x01\x00\x04\x00\x00\x00\x1E\x00\x00\u00FF\u00E1\x03\u0083http://ns.adobe.com/xap/1.0/\x00<?xpacket begin=\"\u00EF\u00BB\u00BF\" id=\"W5M0MpCehiHzreSzNTczkc9d\"?> <x:xmpmeta xmlns:x=\"adobe:ns:meta/\" x:xmptk=\"Adobe XMP Core 5.6-c138 79.159824, 2016/09/14-01:09:01        \"> <rdf:RDF xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\"> <rdf:Description rdf:about=\"\" xmlns:xmpMM=\"http://ns.adobe.com/xap/1.0/mm/\" xmlns:stRef=\"http://ns.adobe.com/xap/1.0/sType/ResourceRef#\" xmlns:xmp=\"http://ns.adobe.com/xap/1.0/\" xmpMM:OriginalDocumentID=\"ED7F53B4B5D49882FFD873EFD5267C52\" xmpMM:DocumentID=\"xmp.did:3929EBFAB89711E6B766BA4A7DF3D973\" xmpMM:InstanceID=\"xmp.iid:3929EBF9B89711E6B766BA4A7DF3D973\" xmp:CreatorTool=\"Adobe Photoshop CC 2015 (Macintosh)\"> <xmpMM:DerivedFrom stRef:instanceID=\"xmp.iid:f5c2cb6d-ff30-47ca-8f14-6d3feb250192\" stRef:documentID=\"adobe:docid:photoshop:fd86a5da-a29d-1178-8fe7-d4b15b34270b\"/> </rdf:Description> </rdf:RDF> </x:xmpmeta> <?xpacket end=\"r\"?>\u00FF\u00ED\x00HPhotoshop 3.0\x008BIM\x04\x04\x00\x00\x00\x00\x00\x0F\x1C\x01Z\x00\x03\x1B%G\x1C\x02\x00\x00\x02\x00\x02\x008BIM\x04%\x00\x00\x00\x00\x00\x10\u00FC\u00E1\x1F\u0089\u00C8\u00B7\u00C9x/4b4\x07Xw\u00EB\u00FF\u00EE\x00\x0EAdobe\x00d\u00C0\x00\x00\x00\x01\u00FF\u00DB\x00\u0084\x00\x10\x0B\x0B\x0B\f\x0B\x10\f\f\x10\x17\x0F\r\x0F\x17\x1B\x14\x10\x10\x14\x1B\x1F\x17\x17\x17\x17\x17\x1F\x1E\x17\x1A\x1A\x1A\x1A\x17\x1E\x1E#%'%#\x1E//33//@@@@@@@@@@@@@@@\x01\x11\x0F\x0F\x11\x13\x11\x15\x12\x12\x15\x14\x11\x14\x11\x14\x1A\x14\x16\x16\x14\x1A&\x1A\x1A\x1C\x1A\x1A&0#\x1E\x1E\x1E\x1E#0+.'''.+550055@@?@@@@@@@@@@@@\u00FF\u00C0\x00\x11\b\x01\x05\x00\u00C8\x03\x01\"\x00\x02\x11\x01\x03\x11\x01\u00FF\u00C4\x00\u0094\x00\x00\x02\x03\x01\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x03\x00\x04\x05\x01\x06\x07\x01\x00\x03\x01\x01\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x03\x00\x04\x05\x10\x00\x01\x04\x00\x04\x04\x04\x04\x05\x02\x04\x07\x01\x00\x00\x00\x01\x00\x11\x02\x03!1\x12\x04AQ\"\x05aq\u0081\x13\u00912B\x06\u00A1\u00B1\u00C1#\x14\u00D13Rbr\x15\u00F0\u00E1\u00F1\u00A2\u00C2C$S\x11\x00\x02\x02\x01\x03\x02\x04\x06\x03\x01\x00\x00\x00\x00\x00\x00\x00\x01\x11\x02!1A\x03q\x12Qa2\x04\u0081\u0091\u00C1\"B\x13\u00D1r3\u0082\u00FF\u00DA\x00\f\x03\x01\x00\x02\x11\x03\x11\x00?\x00\u00D1e\x19u\u0097Yq\u00C1y;\x18:t \u0086\x01:!:B\u00B6\x1C\x020\u00EB\u0091F\x13\u008A\x0B\u0095\x1C\u00A2C)F\x112\u0091\x11\u0088\x0EI\u00C0\x00\u00B1\u0088\u00E5G+\x0B\u00B8\u00FD\u00DD\u00DB\u00F6\u0096\n\u00E9\u00FF\x00\u00E8\u0097\u00D4bZ#\u00D7\u008A\u00F3{\u00DF\u00BC;\u009E\u00E2R\u00F6\u00E6(\u0081\u00C0F\u00BF\u00EAV\u0089\u00D0'\u00BF2l\u00CB.\x1B >\u00B1\u00C3\u008F5\u00F2\u00C9\u00F7M\u00E4\u00FEk\u00A6@.:\u008Ehc\u00BC\u00DC\x01\u0085\u0092\u00E7\u0099[\u00B6\u00C6\u00C1\u00F5lT\u00C5|\u00CA\u009E\u00F1\u00BE\u00AD\u00B4\u00DF0\x06]Ejl\u00FE\u00EE\u00EE\x15\u00CC\x1Bd.\u00836\u0099a\u00F8\u0084\x1Ak`\u009E\u00EB\x15\x18\u00AC\u00AE\u00D9\u00F7\x0E\u00CB|\u00D0'\u00DA\u00B8\u00E1\u00A6G\x02|\n\u00D6@\u00C7\x19F]e\x11\u0080H\x04 \u0090M!\x01\bA\u00939\b\u00AB0\u008AT\x02\u00B1\x00\u00B0\u00C1i\u00C1.\u00C8\u00E0\u009E\u0097`\u00C1c\x19[\u00B8`TN\u00DDG\x02\u00A2]\u00C2W\x01\x10\u0083\u00AE\u0088\u00A6D#\x026v\x11\t\u00D1\u0088@\x028\u00B8L\u0085\x18\"\x110@\u00E5T\u00EE=\u00CA\u008E\u00DDA\u00BA\u00F97\u00F8c\u00C6G\u0090FL\u0086\u00EF\u00B7\u00BBm\u0085\x12\u00BFq&\u0088\x18\x0E2<\u0082\u00F9\u00FF\x00}\u00FB\u009Bu\u00DCe\u00ED@\u00FB[q\u0094\"~o\u00F5*\u00DD\u00E3\u00BCn{\u008D\u00E6\u00CBd\u00D1\x18B\x03(\u0085\u0098\u00C4\u00E2rF\u00AAr\u00FEA\u00D0\x13\"sP\"\u00D1\u00CB\x14\u00DA\u00AB\x12\u00C0\u0084\u00F3\x06\u0081B$\u00A6F\u0099\x12G$\u00E8\u00D6\u00DD%\\\u00AA\u0087#\u00C7\u0097$\u00AE\u00C3$g\x1A\u00A4\x17\x1AC%\u00B1=\u00AC[\u00C5\u009D\u00BF%R\u00FD\u00B8\u0086\x11\u00CB\u00E2\u0087p{J\u0095\u00DB(\u0091\u0089\x04d\u00BDw`\u00FB\u00A2Z\u00A3\u00B5\u00DFKTKF\x16\x1E\x1F\u00EA^J\u00CA\u00CCJ\u0095j\u0084\u00B2\u00C1f\u0093\u00CA\x07\u0091\u00F5\u009DQ!\u00C1px\u00AEj\x0B\u00CB}\u00BF\u00DFI\u00D3\u00B5\u00DDK\u00A4a\t\u009E\x1E\x0B\u00D3\u00B2I\x03P\x13\u0085\u00C5\u00CD%tE\x10\x07\x05b\x01&\x01:(\f0 \u009EH\u00D0M\x13\x14\u00B7\x03\x02\u00A2+\u00C6\x05D\u00A1\x12 \u0099\x18&\b\u0084@\x05N\u00D22\b\u008A1\x14A\u0097pF\r\"77U\u00B6\u00A2w\u00DAt\u00C2\x01\u00C9_6\u00EF\u00DD\u00DE\u00EE\u00E3\u00BA:\u00A4\u00F5\u00C4\u00F4\x0E\x00/I\u00F7\u0087w\u00BA\u0093\u00FC:H\x11\u0090\u00EB\u00E7\u00E4\u00BCKj\u009E)V_\u0092\x1Di\u00D4\x11\x1DE0\u00D6\x074\u00EA\u00A92\x19g\u0097\u0092a\u00A2Y\x00\u00B3\u00B8\u00EA\u00A50\b8\x07M\u0085q'\"\ni\u00DA\u00C8b\u00C9\u00D5Tc\u0084\u0083\u00AD\u00DCn\u00D2WQg \x13\u00E7\u008A\u00BD\u00B6\u00AC\x193u\x0E!.\u00AA\u00F2\u00D3\x16\u00FDJ\u00B0\u00D3\u00AE\x04\u00BB\u00B6_\x16(H\u00C9\x07la\x18\u00E1\u00D4x\u009E_\x15T\u00D4%#\u00F5x\u00E4\u009BY3rNY\x03\u00F9'\u008A\u00F5A\u0084qY\u00B0\u00C4\u0098\u00F6\u00D0\x1D\u00C4\u008B\u009EA,PI[\x1E\u00CD\u0092\u00C1\u009Cr\u00C5X\u00A7\u00B6BG\x18\u00B1<\u00C2Gq\u0095$\u00C5\u0085f\r,Cr^\u00BB\u00ED\u00DE\u00E7\x1Bc\x1D\u009D\u00B22\u0098\u00FE\u00DC\u008F.ES\u0097g\x06'\x0E\x19,\u00E8\x1B6;\u00CE\u0093\u00A6Pc\x12pB\u00B6\u00C8\u00B7\u00A6\x0Fx \u0088A\x06\u00CBq\x1D\u00D6\u00DA\x17\u00C3\x1DC\x16\u00E7\u00C5=\u008A\u00BC(9\u00E4\x11\x16G\x15\x19t%hd\u00C2A$Hd\u0080\u00C5k\u0086\n.\u00D8\u00A2\x06\x18*(\u00C5%Z\x15\u0084Z\x15\u00A0\u00E7)\u00FBl\u0097i\x15W)\u009C\u00A2\t\u00F8+\u00D2\u0082\u00CB\u00EF\u009A\u00A1\u00DB7\x06'OA\x0E<PxM\u0084\u00F9\u00C7w\u00DEY\u00BE\u00DE\u00CEr'N\u00A3\u00A4;\u00B0Uv\u00F4\x19\u00CD\u00D9\u00C0,\x02x\u00A0\u00B99\u00AB\u00DB\n\x01\u00BA'\u0081\u00C7\u00E1\u00FDT]\u00A1`\u00E9U\x0Fo\u00B0\u0091\u00D24\u00B9\u00C5\u0096\u0084;N\u00A8\u0082\u00CD\u00FE\"\u00B46\x14\u00C4\x10H\u00C5iB\u00B8\u0080\u00C0)\u00E5\u0095P\u008F?\u00FE\u00CF\x1C\u0098\u0092\u008A\x1D\u008FQv\x00\u00AFE\u00A4\"\x11L\u0097\u0098g\u00C8\u00C3\u00AF\u00B2\u009A\u0089'\u00A8\u00E4\x02\u00EE\u00E7\u00B4\u00C0\u00D0a\x10\u00C4\u0086e\u00B9\u00A5\u00CB\u00A9\u00A0\x14`\u00D2yhl\rR\x02C\x02\u00DF\u00F1\u00E8V\u00A6\u00DF\u00B7\u009D\x02&&Q\u0096R\x1C\u00FCV\u0094\u00B6\u00D0\u009E,\u009D\bh`\x16\u0087\u00B9\u00A5%\u0082\u0084;E \u00F5\x07\u00E3\u0082\u00B7\x1D\u00B5p\u008B\b\u008C\x13\u00C9BJ\u00D0\u0081-\u0095\u00EC\u00AC,\x0E\u00FB\u00B3x\x0B\u00A22\u00C0\u00F9\x15\u00E8\u00E69*[\u00EA\u0085\u00B4J<\u00C2F\u00B3!\u00D5A\u009B\u00F6\u00AFp\x15n%\u00B3\u00B6M\x19\u009E\u0090r\u00D7\u00E7\u00E2\u00BD\u0086\u0095\u00F3{=\u00DD\u00B5\u00D1\u00BA\u00B2\u00C6$b9\u0082\u00BE\u0081\u00DA\u00B7\u0083}\u00B1\u00ABq\u00F5\x10\u00D3\u00FF\x00P\u00CD_\u008A\u00DB\x1C\u009C\u00B5\u0087#\u008CP\x10\u009D \u0095$\u00D6B\u00D5\u009Ct$\u00AE\u00AE\x14\u0085\x04\u00D8\u00A2\u00EC\u00D4@&\u00A3(\u00C8\u0094]\x04A!d}\u00C7\x00{e\u00A3\u00CB\u00F3[\x0B\x13\u00EE\u008B4\u00ECcX\u00CE\u00C97\u00A0I\u00C9\u00E9fJl\u00BA\u009E\x16U1\x18f\u00EE<\x15\u00FD\u009DX\u00C4\u00B3}G\u00C8\u00E4\x12\u00A5\x10\u00E2$y\u00FA\u00A7\u00D5c\x1C\x171\u00D4lm\u009B\u0087\x05n\x04\u0095Gi b\x19]\u0081\u00E4\u0086\u00E5\x16\u0083\u00A3\u0092$!\u00D1'@:\x11\x06B\x1D\x18\x07\u008A(\u00C1pQ\u00D7\x18\u00B2\u00E8\b\u0080\u00E1\x1C\u0097\bLe\u00C2\x02\u00D0\x04\u00C4\u00CA)6\x02\u00C5Y\u0090J\u009CO\x04\u00AD\f\u0099\u00E5\u00BB\u008D1\u0091\u00B7\u009CK\u0086[?f\u00DCMWRND\x1F\u008A\u00A1\u00BC\u0089\u008E\u00F0\u00C4\u008C%\u009F\u00A8O\u00FB_\u00F6\u00BB\u008D\u00B0\u00C8J9sZ\u008E,\u0088\u00F2\u00AF\u00B5\u009E\u00B0\u00A4\u00C8\u00E2\u009D5^Y\u00AB\u00D8\u0085u8\u00B8WW\n\u0099QrQI(\u0080\re\x14Qt\x13!^w\u00EE\u00A1#\x1A\x00\u00CB\u00A8\u00AFBW\u009C\u00FB\u00A8<\u00A9\u00F1\x04\x00\u00A7\u00CB\u00E8cS\u00D6\u008F3qb\x18\u00E5\u0089(c1\u00AAQ\u00FAc\u0089<\u00F9\x0F\u008A\u00E6\u00E0\u0093\"8\x01\u00F9)\\$\u00D8\u0086'\x13\u00E8\u00B9\u00D1\u00D0kv\u00E9H\u00BCN|\x7F@\u00B5`\x16gn\u00AC\u008CNg2\u00B5\u00E2\x1B1\u009A\u00C9dy\u0084sQ\x19.\u0089.\u009D >e\t\u00B2#<\x13\u00E8\x04\u00C6FX\u0084\u00D8\u0090\u00AAF\u00FA\u00DD\u009D4[\x1E\x07\x04SFh\u00B0He\u00C1 \u0095\x19\u00EA\u00C9\x0F\u00B9\u00A6:\u008E\x00f\u008C\u0082\x0B\x1A\u0090\u0092\x1F\x1C\x15)w\x1AL\u00B4W!9\u009F\u00A68\u009F\u00C1\"\u00CD\u00D5\u00A7\x02Dq\u00C9\u00C3\u00FE\u00A9]\u00BE!K\u00C7\x06\u009E\u00B8\x1C\x14\u0094_\x10\u00B3F\u00E6\u00F8\u00FD8s\x00\u00CB\u00FA\"\x1B\u00CBm\x7Fna\u00F2 \u0083\u0087\u0098[\u00BB\x19@\u008F\x06S\u00EFp\u00F6n\u00AFr~S\u00D3 \u00E9\x1D\u00AFq\n;\u00ADv\x13\u00D2KK\u00CAY\x14\u00CE\u00EAws\u00A3M\u00A2\x16\u00C0\u0097\x11\u008FL\u00B0\x0F\u0082\u00C5\u00ED;\u008D\u00B6\u00F3\u00B9T*\u00A6B\u00C6\u00D3\rV\x12\u00C4x`\u0095e\u00B6\u00B6\x05\u0096!\u00EE}\"y$\u00C8'\u0091\u00D2\x18\u00BE\x1C\x12\u00E4\x17C9\x10\u0082\u00B8We\u0099\\*e@\u0092\u008AID\fk(\u00A2\u00E2\u00E8&B\u00B1>\u00E6\u00A8KoU\u0087\x03\x19\x10\u00FEam\u00AC\u00BE\u00F8\u00D6\u00EC\u00EC\u00A2#T\u00C8\x04x\x14\u009C\u0091\u00DA\u00E44O\u00B9B\u009C\u009E\x16e\u00ED\x11\u00E1\u00C7\u00D1Z2\x1AD\u00F4\u00B4y\u009C\u00CB~\u0081W\u009Fj\u00DE\r\u00C4,\u00BB\u00A2\u00B7x\u00E98\u0094\u00FBv7K\x01#*Aa\x07c\u008E`2\u00E7\u00C1\u00D5\r2\u00DC;\u00AE\u00C3gP\u009D\u00B38\u0092\x00\x00\u0092S)\u00EF\u00D3\u00DE\u00E1\u00B0\u00A7\\F\x06s\u0090\x00z\x05Gq\u00F6\u00FD\x1B\u008D\u00B8\u0085D\u00D5o\x03\"e\x12\u00FC<\x15\u00CE\u00C7\u00DB\u00E7\u00B0\u00ED\u00E6\u00A9\u00E9\u00D4&I\u0093\u00F3YDk\u0090\u00B9\u009C\u00A4X\u00B3wl\x07\u00EEY\u00D5\u0099\u008Dqa\u00EAd\u00B3\u00ED\u00EEq.a+\x0B\f@8~\x00+cge\u00D3\u0094\u00DC\x18\u00C9\u00F4\u00C4\u008ExjI\u00AF\u00B1\u00DF\x0Bclt\u00BCb`\u00ED\u0081\x07\u008B\x10\u009E\u0094\u00AB\u00F56-\u00EDd\u00FE\u00D4\u0099\u00C39\u00FBbb\x03\u00AA.\x04\u00A57/\u00EA\u00BB\r\u00C4g\x1D&F\u00B9\u009C\x03\x13\u009F\u00C5Y\u00BBilv\u0095\u00ED\u00CB\x01Xa\"\x1F\u00F3T\u008D\x15\u00C1\u00A2\u00E6s\u00E1#\u00C1k*\u00AD\x10j\u00EC\u00F2\u00DF\u00C0\u00A3\u00FE\u00F3\u00DE\u00E8\u00DFKgI3\u00B0\x1E\u0098\u0080&\u00E0\u00F1\u00EAV;\u009E\u00E3\u00EE\b\u00ED\u00FD\u00CB\u00A1/i\u009E\u00C6!\u0087\u0098\u008B+\x1D\u0088B\u00FE\u00E5\u00B9\u00DC\x7F\u00F9\u0081X><W\u00A4\u0094#d%\t\u0081(H\x11!\u00CC\x14>\u0083}O+\u00DB+\u00B0\u00EC\u00A1p$\x1BF\u00A9\u00CF&\x0F\u0090d\u00D9\x19\n\u00E3*\u00A5\u00A4JM,_\x0F\x15g\u00B6\u00C0\u00EDgwm\u00B07\u00B5#*\t\u00FA\u00AB\u0096!\u00BC\u0096\u0085;z\u00A2_K\u00F3\u00E0Q\u00ABS\u00E2\x0B)Mi\u00D0\u00C7\u00A2\x16Ym:\u00A5`\x1F\u00FBt\u00BB\x11\u00CF2\u00B4k\u00AA\u00EFsU\u00C4h\x18W6\u00EBo\u00F32\u00D2\x1E\u00D30\r\u00E6\x10[\x1E$\u0088\u008Eg\x04n\u00D3\u00DA:\t\u00C6\u00BBTe\u00F5+n\u00EB\x13\u00A8\x18\u0090tJ'\u00F1Y\u009D\u00BFa\u00B1\u00DB\u00EEee\x11\x1DN5\u00E7\u00E6\u00CBR\u00EBg:\u00E5V\u00D6\x1E\u00E4\u00E4\x1BY\u00C2\u00B8x\u0099q\u00F4T\u00F6\u00FB_\u00E1\u00CA\x14\u0083\u00AA1n\u00AEg\u0089\u00F8\u00A9Yh\u00CA,|\u00CD\u00EE\u00D5v\u00B8\u00CE\u00A2\u00ED\x03\u0083\u00AB\u0093\n\u008E\u00C0\u008F\u00E4\x120\u00D4?E~y+\u00F1\u00FAz\x1C\u00DC\u00E9.G\x1B\u00C3*H\u00F5\x14%v\x7F1BR\u00B0-\x01*.\x12\u00A2\x066\x14Qqt\bB\u00BC\u00DFs\u00B6\u00E1\u00BF\u00B69\u00C1\u00C6\x1E\u008B\u00D1\u0092\u00B1;\u009Dq\x1B\u00D8\u00CC\u00E5 \t\u00F4Q\u00E6SU\u00D4\u00B7\u00B6k\u00F6?:\u00B32\u00D8\x19\x003/\u0082\u00E4\x00\u00C0\"\u00DEn\u00AB\u0085\u00B5\u00D6b\u00C6\u00E3\u00A472\u00B8j\u0095w\x11\u00C02\u00E7\u00D0\u00E9jYn1\u008C\u0080\u00C1\u00CA\x01Q\u00A6\u00EB&^T\u00DA\u00C5\u0080}2\x19\u00E0\u008E\x07.\x03\u008Atd\u00ECy\u00FE\x012\u00CF\u00C0\f\u00AE'\u00B6\u00E1`\x1E\x04\u00B7\u00E0SF\u00E6\u0098\u008F\u009E>\u00B2\x1F\u00A2a\x10\u0096a\u00C7\u0092\u00E0\u00AA\x03\x11X~\x18\x04\u00EAA\bE\u00B6\u00C2\u00D8\u009F\u00DC\x12n\x11\x04\u00AC\u00DB\u00EB\u0094\u00E1*\u00F6\u00F1\"\u00C9\u00E0l\u009E\f<\u00B3[\x13\u00AES\u00C0\u0096\x1E\tR\u0084*\u00F3\u00FCR\u00DAXj\u0092)v\u00CD\u009C65\u008A\u00A0L\u00A5\u0089\u009C\u008Er\u0091\u00CC\u0095\u00B1\x13\u00D3\u0081T\u00AAb\f\u0088\u00C4\u0095r\u00B8\u00C8\u00C5\u00F8&\u00A8X\u008D\u00C6\u00DA\u00BB\u00A5\x19\u0091\u00FB\u0090\u00F9NG\u00E2\u008E de y\x1F\u00FA#v>!\x1Cg\x02ZX\x1E\bB\u0099\x03\u00F0\u00D4\f\x07\u00D6J\u00EF\u00B7\x03\u0089\x1A\u00BC\u00F1\u00FC\u00D3@\x1E\u008B\u00B8pF\x05\u00E8\x01\f\x1F\u0082\u00AB\u00ED\u0089n5\u009C\u0080b<\u00D5\u00A9d\u00C9\x0E\x00\u00B0\u009E\b0\u00A1\u00FB.\u009D\u00CBp\u00C7\u00F2W\u00E6pY\u00F4K\u00F7\u00E1,\u009F\u00FA+\u0096K\x04\u00FCx\u00AB\u00EAs\u00FB\u008FZ\u00FE\u00A8D\u00CFQ@J\u0084\u00E2\u00B8J\x02\u00A3\u0085E\u00C2TX&\u00CA\u00E1+\u008E\u0086RW%'IY}\u00D8u\u00D5/\x02\x16\u0081\u0092\u00A1\u00DC\u00B1\u00AA3\x18\u00E8?\u0081S\u00E4\u00CDX\u00FC6\u008EZ\u00F9\u00E3\u00E6b\u00D9\x11>\u00ED@8\u0088\u0089Hy\u0080\u009F\u00B9\u00F9\u00DCq\\;['`\u00DDBm\u00EDc\u00A5\u00BE`FK\u0093\u0090\u00B2\x02A\u00C3\u008C\u00B8\u0085\u00CFc\u00BB\u00C3\u00CB\x07k\u00911nx\x15b8c\u00CC\u00AA\u00B5\u00E0\u00CA\u00C8!\u00BC\u0096\u00AB\x03\x1F\x02\x13\x01\x05U\u008C\u00F0L\x13\u00D3\x17\u00E2\u00AA\u0098\x1A\x0E\u00C9\u0088\u008C3T\u008C\u00BF\x7F\u00AB\"\x15\u0082_\u00A8\u00AA\u00D7\u0081,q\x07\u0081\x1C\x10aD3\u008C\x0E\x1C\u00F2Wj\u00DC\u0081\u00A48b\u00B1%\u00B1\u008D\u00B6\u00FB\u0096<\u00A42.\u00CC\u00AC\u00C6\u009B\u00830$\f\u008A\nPl\u0093\u00DC\u00BDv\u00E6\u0098\\!9F2\u009F\u00C9\x12X\u009F \u008FLf\x1C\u00FA\x15Z[8Z\x04\u00AD\u00882\x18\u0082s\t\u00F1\x061\x00d\x11\x06#\f:\u00EC\u0094\x0E\u0089\x1F\"\u009D\u00A9\u00C2C\tE\u008Ek\u0090\u0099\x12\u00D1,\u00F8\x1EkH\x1A\u0091\u00B2\u0092\bW\u00EEI\u00DD\u0080\u0096\\\u00F0\\\u0096\x12\x0F\u00C52\u00994e\u00CD\u00D0\u00C3\u00D4\x19\u00D8db\x05\u0091<\u0091NhC\u0080\u00E72\u0082E>\u0088\u00E6\u00BF\u00DDn\u0098!+\u008E\u0085\u00D7\t@0t\u0095\x10\x12\u00A2\x06\u0083d\u0094\u00B9\u00C9\x1C\u008E\t\x13\u0092\u00BB9\u00D9\u00C9MU\u00BE\u00D8\u00E9\"Y\x14vI\u0096n\u00F6\u00E2\"T\u00EC#\u00B3YCc`\u0095mQ\x1C\u008F\u009A]\u00C3\x10W\u009D\u0097q\u00BFm\u00B83\u00AAL\u00F8\x18\u009C\u008A\u00D2\u00AB\u00BCQt\x1A`\u00C2|x\u0085&\u00B0z\x1C\\\u009D\u00C97\u0086Z\u0089\u00C4\x0F\x14\u00F7\u00E5\u00C9U\u00A6\u00C8\u00CC\tD\u00B8\u00C9\u00D5\u00AA\u0087W\u00A2J\u00EAU\u009C\u00AB\x10\x11Jz\u00A7\u00A5\u00F0\x19\u00A9C8\r\u00C1\b\u008BFRg.\u00E5:\bFn9\x04:uz\u00AC\u00FF\x00\u00F7]\u00A8\u0099\u008C\u00AC\x00\u00FF\x00\u0085\u00C0?\x04\u00F8o\u00A3,`\x1D\u00BDS\x1A\x0B\u0091\u00AD\u008Ax\u0088a\u00E2\u00A8\r\u00C5\u0084k\u00D3-\x0E\u00DA\u0099:Wn+\x02F\u00A9\u00E3\u0091!\x14\b\u00D1\x16\u00C8\u00C3\x0486x\u00E6\u0096a\u00BC\u0098\u008E\u00AA\u00C4u\u00E5\u008B\u009F\u00C1.\u00ED\u00AE\u00E6\x15\u00DB/s\u00FBa\u00E2\x00\u00F9\u008A\u00CC\x12\u00BCP\u00D3d`\x1C\u0094\x1A\u00E3t\x04\u00E3\u00F4\u0097\u0089X\x1D\u00BE\u008E\u00E5\u00DCw\u00F6[\u00B8\u00B6Q\u00D9W\u00F2W\x1C\x04\u00CF\u00E6\u00CB\u00D1UN\u008E\u0081\u0084Pk`\u00F9\u0087 \u00F2\u0081G\u00B7\x0F\u00AC\u00F28!\u009F\u00F7\x07\u00F9Bf\u00CC=s<\u00E4\x7F\x00\u0095\u00E1\u0089oHRJ\u0092\u00B1(\u00A4\u00CC,\u00ACK\u00B4Q+\u0084\u00AEH\u00E2\u0084\u0094d\u00D0t\u0095\x10\x12\u00A2\u00D2h6\u00A7,\x12&S$R\u00A4\u00BANh+^K,\u008D\u00E9$\x15\u00AFp\u00C1e\u00EF\x06\x05%\u0085\u00ED\u0093\u00CDn\u00C3M\rRd\u00DD\u00D8{\nP\u008B(\u00D8\u00E9\u00E3P\u008D~\u00D7\u00B8yJ\u00B3\u00C7\x10\u00B5\u00EB\u009FP+\u00CDm$ch#\u0082\u00DC\u00AA\u00D7\fT\u00E7'E2\u008B\u00B5\x1D6\x1C0t\u00E9@1#\u008Ej\u00A1\u009ER\x07\u00CD<Z%\x07uD\u00C2e\u00F7\x0E\u00CD\u00B7\u00DCL[\x10!lK\u00C6m\u009F\u009A\u00EFn\u008D\u00BBI\u00E9\u009C\x01\u00AFK\x13\x1CN\x07\x07\x05iH\x03\x12\u0093\u00A0\u0083\u0082-\u00EC\u00C2\u0095\\\u00CA\x0E=\u00C3kUVBQ\u0090\x00\u0091\\t\u0092\u00EF\u00C7\u00C9]\u009E\u00E3m*ZS\x06!\u008E\x19\u00E1\u008A\u00A4\x04Kj\u008E)\u0091\x15\f4\u00A2\u00BA\u0099\u00F1\u00D7_\u00B8-\u00E6\u00F6b\u00B8\u00FF\x00\x0F\u00AE\u00E1 b\u00E1\u00A0\u00C31/4\u00BB'~\u00E6\x04Z\x04#,Lb\u00FF\x00\x07N\x1ANPe\x1B\x0E^\x0B\x19*WE\u009F\x17\u0090v\u00F5\u008A\u00E2\u00D1\u00C2#$\u00D8\u00C9\u00A4\u00E7&B\x02\x1Bd@\u00C3\u008E\x00 \u00D8\x1EY\u00C9\u00D8\x04g?\u0082\u00B7\u00B2\u0089\u008E\u00D6\x0F\u009C\u009EG\u00D5P\u009C}\u00CB+\u00DB\u008F\u00A8\u00B1>\x1CV\u00B3\x00\x00\x19\f\x02\u009D\u009EE\u00B6\u0089\x01%^\u00C5bJ\u00BD\u0088!\n\u00F28\u00A0%vY\u00A1%1\u008E\x12\u00A2\u00E1*,h5\u00F5\x02\u0082E\x07\u00B8\x14\u0094\u00C2\u00EBg<\n\u00B4\u00E0\u00B37\u0098\u0082\u00B4l\u0090Y\u00DB\u00A2\u00EE\u00A7`\u00A4an!\u00D6\u0096`\u00AC\u00DF\x1E\u00B4\x06*\x17y-U\u00806\u00E1\u00AC\x0BT\u00D7/lY\x1C\u0087\u00CC?U\u009BH\u00FD\u00D0\u00B76\u00A0\x18i9\x10\u00CAM\u00E4\u00A5\\\x0B\u00A2\u00D0b\u00C7\"\u009B\\\u00CC&br9*\u0092\x06\u00A9\u00903\x05\u0088M\u008D\u0082c\u00C6*\u0089\u008E\u00CB\u00BA\u00B1]\u00CF$\u00BA\u00E4$\x07\x12\u009B\x10\u0098\u00C8\u0087P\u00CB\u00E2\u00A6\u00B9a\u0086\b\u00DF\x04P\u00AD\u00F3\u00CD27P^d\u00F8x#\x11%\x16\u0086\x1E+\u00A1\u00F2Y\u0080\x02\u00E1&\u00C9\u0081'\u00E0\x13\u00E7 \x01T\u008C\u00BD\u00C908q\u00FE\u0089X\x02\x06E\u00EC\u0089i\u00FD'\u0092\u00A5\u00D9\u00BE\u00EC;\u008D\u00EC\u00FBop\u0088\u0086\u00E2\x121\u008D\u0083\x01&<B\u00D0\x01\u0081^+\u00BBPv\u00FF\x00s\u00D1d0\u00F7\u00A5\tz\u00FC\u00B2M\u00C2\u00ABk\u00F6\u00D9z\u0085\u00E7MUY~'\u00D2$\u00AB\u00D8\u00BB\u00B6\u00B7] \x13\u00D40\\\u00B1N\u00D5t\u00BB\u00AB\u00D9\u008BW)5\u00B9Ry\u00A0%\x1C\u00F3K+\x04\u00E1Qp\u00A8\u0089\u0086Gt\x0F\x14~\u00FE\x19\u00AF5Gqq\u009A\u00BB\x1D\u00E8#5\u00D7c\u009D\x1As\u00B9S\u00BE\u00C7I\u00FES\u00F1J\u00B2\u00D7R\u00B3\x19!Vc$2\x18(e\u008A\u00E9\u00C9s\u00DBR\u00A8\x1A\u0083X\u00EBkhpX\u00B0\u00FE\u00E0[\x1B<\u0082\u009B\x1D\x13y[Z\u00FC&\x1F\u00D4`UR%\x12\u00F1\u00C0\u008C\u00D6\u0096\u00F2\x02Tj\u00E3\x02\u00FE\u009CU\x00\t.\u00CA\u0090\u00D4H\u00F5j\u00D5\u00E8J\u00B7\r\u0083\u00B1\u00E2\x15\u00D8]\x12\u00D8\u00E6\u00B3\u00EC\u00AB\u008A\b\u00DBeg\u0098T\u008F\x01d\u00DA\u0085\u00B1N\u008C\u00A2\u00D8,xn\u00C1\u00CF\x02\u00ACG{\x06g\u00C923h\u00D1\u00D6\x01C+@\x0E\u00B3\u00CE\u00FA',J\u00E1\u00B6\u00EB>P\u00C0\u00E0\u00E5f\u00C0:\u00FB\u008C\u008F\u00B7\f\u00CE~K\u00B4\u00C0\x00\u00DCW)\u00AB\u00DB\x0Es\u00E2S\"\u00D9\u00A9\u00B6=C#\u00A7\x15\u00E6\u00FB\u00DE\u00DB\u00DC\u00EF\x1D\u00BAC\x13\u00AAO\u00E4:\u0097\u00A4\u0099a\u008F\u00AA\u00C7\x03\u00F9]\u00D8Lc\r\u00B4N?\u00E6\u009F\u00FC\u00958\x14\u00F3R6bs8\u00E2\u00B1\u00A9M\u0086\u00B29+s\u0090\u0094\\dVy8\u0094R\u00B6\u00C8\u00D4\u00F5\u009E\u00A8\u00E2\u00DC\u00C2\u00EC\u00E7\u00F6\u00EB\u0092\x1Ap\u00CE\x1E>^\u00D9O(;3K+\u0095oi\u00BB\t\u0086\u0090\u00CD;\u00DA\u00AE\x7F$\u00FD\n\u00E5\u00BF\u00B6\u00E4\u00AE\u00D2^\u00BC\u00D4\u00B6\u00E2\tQ2[{G\x07\u00F2QK\u00B2\u00F3\x1D\u00AF\u00E4?uu\u0094|\u00F3o}\u0080\x05\u00A3M\u00F3!.\u00BD\u0099\x03%n\u008D\u008D\u00D60\u0084I]/$V\x03\u00AA\u00D3\u0086)\u00B2\u0099V\u00F6\u00FD\u0096\u00CC\r\u00D2\x10\x1C\u0086j\u00F46[:q\u00D3\u00AEC\u0089C\u00F4\u00DE\u00DBGP\u00FE\u00CA\u00A3.\u00AAn\u00B5\u00B4@\u0095r\x1D\u00B2\u00E9\x0E\u00B2 \x15\u00D3v\u0091\u00D2\x00\x01&\u00DD\u00C4\u00F4\u00E6\u009E\u00BE\u00CD?S\u0091_<hr\x1D\u00BFkI\u00D5d\u008C\u008F\u00E0\u0099\u00FC\u009Aa\u00D3_\u00C0,\u00DB/\u009C\u00DD\u00CE#\u0082=\u0090\x12\u0099\u0094\u00B8.\u009A\u00FBjQL\"/\u009A\u00D6q#\u00EC\u00DD\u00F54\u0086\x04\u00B6>(\u00E0\\\u0096\u00E0YV\u00B6\u0099]\u00BE\u00A8\x0F\u00ED\u00C4\u00EA\u0097\u00A2`\u009F\u00B5\u00BC\u0095g\u00E5\u00BB\x18\u00FF\x00\u00A8f\x14=\u00D7\x1AtVZ\u00D7?\x02\u00FE\u00DB\u0092.\u00EA\u00F4\u00B6>%\u00A8\u0080C\x15\u00D1\u00B7\u0084\u00B8 p\nmsb\u00B8jv4\x01\u00D8W/\x02\u00A4{|8\u00AB\u0090\u009CH\b\u008B\x01\u0082\u00A4\t\u0092\u00B8\u00DBW\x06`\x13DG\u0093.\u0092\x0Ehfp>H0\u00A4rD\u0092\x00F:\x03\u00FA%\u00C1\u00C1yg\u00C1r\u00DB#\b\u00B9*~e\x05owb\u009A\u00A7<\u00C3`9\u0094\u009E\u00D9\u00B7\u0095;se\u009F\u00DC\u00B4\u00EB\u0099\u00F1*\u00A4L\u00F7\u00DB\u00A1\f\u00EA\u00AC\u00B9~%jXDF\u0081\u00C1w\u00FB>(]\u00EF[at8\u00BD\u00DF.{\x17\u00E3\u00AFRf\u0086Y)\x1C\u00B1P\u00F1]\u0091\u0093\u0092Jt\u00C3\u00AE@\fNj\u00E4*.\t,\u00CAm\u00AA\x11\u008E\u00A2:\u00A5\u00C52e\u0083qF\u00CEX\x12\u0084\x07\u00BD8\u00CFL$YE\u00C8\u00C4G\x12\u00A2\x1D\u00AB\u00C02\u00CA\u00F4v\u00CA\u00E3\u00D5g\u00A4U\u00A1\u00A2\x01\u00A2\x04G\u0082\u00ED\u00B3l\x02\u00ACI9\u00A9S\u008D$=\u00AE8\u00D8\x12\u00E4I\u00F2@\u00C8\u0086\x18\x15N\u00D4\u0085\u0099\x03Q\x18\x1C\u0095{\u00A4@\u00C3\x14\u00FB\x01\fx$\u00CE.|\x13U\n\u00CAe\u00F5\u00B9\u00E2\u00ADm\u00C1\u008B\u008E\x05(\u00D5\u00AAC\u00C0\u00AB\u0095\u0088\u0082\u00FC\x026x\x05\x10\u00CD\u00BDZI\u00B3\u0089\u00C1+}A\u00B6\u0097\u0089k uBC\u0081\t\u00A6\u00E80\x11.\x1D\u0089\u00E0\u00884\u00BDT\u00A2u\u00DC\u00A2q\u00A6\u00C5-\u009E\u00F0n+b4\u00DB\x1E\u0099\u00C3\u0094\u0082\u00B1\x19\u0090[\u008A\u00CB\u00EE4\u00D9\u00B3\u00DC\x7F2\u009F\u0094\u00B0\u00B4\x0E#\u009A\u00B7\u00B6\u00DD\u00D7p\x04\x1C\u00D7\u009B\u00CD\u00C4\u00F8\u00ED\u008D6=\x1E.Uz\u00F9\u00EE_\u0085\u00C4\x15b6\u0099\x06U\u00EA\x02J\u00CC\x03`\x10\x18-%\x04\u00FAC\u00A3\u00D4\u00CA\u00AD\u00F3&Y\u00B4x\u00A5fAkq\u0089uC}\u00B9`!\ffp\x01\x1D\u00DB\u009Fn%\u008EI}\u00BBm+\u00EE;\u009B1\x1FK\u00A6\u00E2\u00E3|\u0096UZn\u00FC\u0085\u00E5\u00BFe[z\u00EC]\u00D8\u00ED\u0086\u00DE\u0097?9\u00C6G\u00C5t\u009DR%2\u00E9\u00FD!*9\u00AFZ\u0095Ic\u00E0y\u0096\u00B3o!\u008C\u0091D:\x10\x11\u00C4\u00B6E\u008F\u008A,\b/\u0094%\u00CANp]\u009D\u0098\x17\r\u00F9$\x19\x06\u00C0\u00A0\u0090Y\u00D9\u00DA\u00D8(\u0080@\x12\u00A2l\x0B\u0091\u0096\u009E\u00A49\u00BA\x0B\u00A5cj\x7F\u0082^\u00A2q\u00D5\u009A\n\u00B8\x0B\u00B6F\x02\u00CB\u00A6\u00D8\f\u00C8\t&#\u00C4\u00A1 \x0E\t\u00BBE\u00EE\u0081\u00E6~\u00E4:\x07O2\u0091f\u00A8\u00E2\u00F8&\u00CC\u0098W\x18\f\u00F8\u00AEn+:!\b\u0087\u00E6\u0082\u0084\x16\u00D9Lk\u0094\u00D8\x12\u00AC\u008A\x1FLs'2S)\u00A0V5\x1CJ\u00ED\x0F+I\u00E0\x16\u00B5\u00BC65k\x1A\u00EEvQ\u0088\u00FD\u00B02]\u00AA\\\nU\u00B6i\u00DC\u00C4\x1C\u0089c\u00EA\u008F\u00E5\u0091I\x18\x1C\u00B1e\x11\u00B2&2\x0E\x0Eay\u00BD\u00CE\u00DA\u00FE\u00D9\u00B8`\u00FE\u00C4\u00CF\u00ED\u00CB\u00FF\x00\x12\u00B5;\u00BFy\x1D\u00B3i\x13\x10'\u00B8\u00B4\u0091TNA\u00B3\u0091\u00F2X\u00BB\x1E\u00E5\u00B9\u00DEZa\u00BE\u0099\u00BE\x16\u00E1\u00A0\u0081\u00A5\u00DF\u0080\x19$\u00B7\x17\u00ECQ\u00E0=y;\x1Fv\u00C6\u00CE\u00D3z\fG5\u00A1\r\u00D4H\u00C5eC\u00B7\u00C6\u00B3\u00FBV\x18\u0087\u00CA\\=V\u0085}\u00B6\u00FD/\u00EFV\u00DC\u00F5:\u00E2\u00BF\r\u00EA\u00F2\u0099\u00D7NjYj\u0086J\u00F1\u00CF\x05V\u00EBIr\x15\u00D8v\u00FAF6\u00DF\u00AB\u00C2\x03\u00F5)\u00A4\u00ED\u00B6\u00F0\u0094\u00EB\u0080\u0080\u0088s9bX$\u00FDV\u00B64\x19\u00F3V\u00B9\u00D4\u00C7\u00FE%\u00F6\u00CA\x1A\u00E2c\t\u009E8:\u00D6\u008CcEB#\x05\u009F\u00B2\u00EF\u009B-\u00D6\u00EC\u00D3a\u0094/%\u00A0g\u0094\u00FC\"U\u00CB\u00E6\u00E7\u00C9w\u00F0p~\u00B5\u00DB\u00BF\u00E4qss;\u00BE\u00ED\u00B6\x00\u00C9\u00CB\u00A6B%*\x03Q\n\u00CCC\x05w\u0082(\u0082(b:\u008B\u00F1]\u009D\u00C221\x01\u00C8\u00CD&V\u00C9\u00F0\bda\u00FACrUm\u009C\\\u00C6!\u00CA9{\u00B3\f\u00EC\u00E8\u00AA\u00DB\u0088\u00E7\u008A\x0B\x06y9M\x18<\u0086*+@\x00\u00A2\x1D\u00CEd\u00D1\u0082\u008C\u00DBF9*\u00F5e\u008E\\\x14QR\u00BA\tm\u0083(%\u00F3E\u00F2|TQ0\u00ACd\u00FF\x00\u00B85'O \u00EA(\u0091\u00ECQnp\u00B6\u0095(\u00D3\u00D4\u00CA(\u0091\u00E8\u00C6\u00DC\u00A1\u00BC\u00D5\u00EFE\u00BEg\n\u00DC\u00BEo\x16\u00C5E\x13[J\u008B]ly\u00FF\x00\u00BA\u00FD\u00AF\u00FEg'\u00DDii\x1C4\u00BF\x15k\u00EC\u008Fg\u00F9\u00D6{\u009A}\u00DFo\u00F6\u00B5s~\u00AD>\u008A(\u00A7\x7FE\u00BF\u00AE\u00C3\u00AF\u00C7\u00FB\x1E\u009B\u00BD\u00FBMV\u0096\u00F7\u00F5`\u00D9\u00E8\u00FA\u00B5,\u00D8\u00EA\u00C1\u009F<\x19E\x16\u00E3\u00FF\x00/\u00E7\u00E8\x0Bz\u00FF\x00\u0082\u00D42\u00CE_\x00\u00B3\u00FB\u00D7\u00F2=\u0098\u00B3\u00FB\\t\u00BF\u00FD\u00CA(\u0087\u00B7\u00FF\x00M\u00BF\u00EB\u00E8noC\u00D7\u00E0y]\u00C6\u00AFv\u00BFm\u00FD\u00CDq\u00D0\u00D9\u00BB\u00AFk\u008F\u00D5\u009F\x1F5\x14]\x16\u00F5\u00D8\u0095}\x15\x1D^\u0096\u00C16*(\u0091\u0094E!\u00EE\u00FB\u00A5\u00B1\u00C4\u00BE\u00AC?%`7\x1C\u00D4Q5\u00B6\x16\u00BB\u0087\x16t\u00C1\u0092\u008A)\u00B1\u00D1\u00D0\u00A2\u008A,c\u00FF\u00D9" ;
		var myWindow = new Window("dialog", "About Keith Gilbert", undefined, {closebutton:false});
			myWindow.alignChildren = "right";
			var myTextAndImageGroup = myWindow.add("group");
				myTextAndImageGroup.orientation = "row";
				myTextAndImageGroup.alignChildren = "top";
				myTextAndImageGroup.margins = 20;
				myTextAndImageGroup.spacing = 40;
				var myTextGroup = myTextAndImageGroup.add("group");
					myTextGroup.orientation = "column";
					myTextGroup.alignChildren = "left";
					var mySellText = myTextGroup.add("statictext", undefined, "Has this script saved you time? You can show your appreciation by buying me a cup of coffee here:\r\r"+
                        "https:\/\/www.buymeacoffee.com/keithgilbert\r\r"+
						"Do you want to automate your specific workflow? For a fee, I can write customized scripts like this one for your unique needs.\r\r"+
						"Need custom training or expert advice? I do training and consulting for a wide variety of large and small businesses.\r\r"+
						"I'd love to talk with you about your needs and requirements. You can find me here:\r\r"+
						"     Web site: www.gilbertconsulting.com\r\r     Email: kgilbert@gilbertconsulting.com\r\r     Twitter: @gilbertconsult\r\r     LinkedIn: www.linkedin.com/in/keithgilbert1\r\r",{multiline:true});
						mySellText.minimumSize = [300,undefined];
				var myImageGroup = myTextAndImageGroup.add("group");
					myImageGroup.add ("image", undefined, myHeadShot);
					myImageGroup.margins = 0;
			var myButtonGroup = myWindow.add ("group");
				myButtonGroup.orientation = "row";
				myButtonGroup.alignChildren = "center";
				myButtonGroup.margins = 20;
				myButtonGroup.spacing = 40;	
				var myHideField = myButtonGroup.add("checkbox", undefined, "\u00A0Hide this box for 30 days");
					myHideField.value = false;
				var myExportBtn = myButtonGroup.add ("button", undefined, "OK", {name:"ok"});
		if (myWindow.show() == 1) { // User didn't click the cancel button
			if (myHideField.value == true) {
				var logFile = File(myTrimPath($.fileName)+"/create_angled_table_heads_required.txt");
				var today = new Date();
				var myTimeOffset = today.getTime();
				logFile.open("w");
				logFile.write(String(myTimeOffset));
				logFile.close();
			}
			return;
		}
		return;
	}
}
// Trim the characters from a file path, starting from the right, through the first "/" character 
function myTrimPath(myLongPath) {
	var myString = myLongPath.toString();
	var myLastSlash = myString.lastIndexOf("/");
	var myPathName = myString.slice(0,myLastSlash);
	return myPathName;
}
// Find a paragraph style possibly contained inside a style group. Return id of the paragraph style.
function locateParagraphStyle(styleNameToLocate, pathSeparator) {
	var myDoc = app.activeDocument;
    var rootPath = "";
    if (pathSeparator !== undefined) {
        rootPath = pathSeparator;
    }
	var retVal = locateParagraphStyleRecursive(myDoc, styleNameToLocate, pathSeparator, rootPath);
	if (! retVal) {
		return null;
	}
	return retVal.id;
}
// Find paragraph styles that are in style groups. Returns the paragraph style or null if not found.
function locateParagraphStyleRecursive(paragraphStyleContainer, styleNameToLocate, pathSeparator, parentPath) {
	var retVal = null;
	do { 
		try {
			var paragraphStyles = paragraphStyleContainer.paragraphStyles;
			var numParagraphStyles = paragraphStyles.length; 
			for (
				var paragraphStyleIdx = 0; 
				retVal == null && paragraphStyleIdx < numParagraphStyles;
				paragraphStyleIdx++) {
				var paragraphStyle = paragraphStyles.item(paragraphStyleIdx);
				if (paragraphStyle.isValid) {
					var paragraphStyleName = paragraphStyle.name;
                      if (pathSeparator) {
                          paragraphStyleName = parentPath + paragraphStyleName;
                      }
					if (paragraphStyleName == styleNameToLocate) {
						retVal = paragraphStyle;
					}
				}
			}
			if (retVal) {
				break;
			}
			var paragraphStyleGroups = paragraphStyleContainer.paragraphStyleGroups;
			var numParagraphStyleGroups = paragraphStyleGroups.length;
			for (
				var paragraphStyleGroupIdx = 0; 
				retVal == null && paragraphStyleGroupIdx < numParagraphStyleGroups;
				paragraphStyleGroupIdx++) {
				var paragraphStyleGroup = paragraphStyleGroups.item(paragraphStyleGroupIdx);
				if (paragraphStyleGroup.isValid) {
					var path = parentPath;
                      if (pathSeparator) {
                          var paragraphStyleGroupName = paragraphStyleGroup.name;
                          path += paragraphStyleGroupName + pathSeparator;
                      }
					retVal = locateParagraphStyleRecursive(paragraphStyleGroup, styleNameToLocate, pathSeparator, path);
				}
			}
		}
		catch (err) {
		}
	}
	while (false);
	return retVal;
}